unit Unit1;

////////////////////////////////////////////////////////////////////////////////
// (c) 2017  Robert Rozee
////////////////////////////////////////////////////////////////////////////////
interface

uses
  Windows, Messages, SysUtils, Classes, Graphics, Controls, Forms, Dialogs,
  StdCtrls, ExtCtrls, Buttons, Math, ComCtrls, ClipBrd, Menus, Registry;

type
  TForm1 = class(TForm)
    Timer1: TTimer;
    Timer2: TTimer;
    Timer3: TTimer;
    Timer4: TTimer;
    Cursor: TLabel;

    Label1: TLabel;
    Label2: TLabel;
    Label3: TLabel;
    Label4: TLabel;
    Label5: TLabel;
    Label6: TLabel;
    Label7: TLabel;
    Label8: TLabel;
    Label9: TLabel;
    Label10: TLabel;
    Label11: TLabel;
    Image1: TImage;
    Image2: TImage;
    Memo1: TMemo;

    OpenDialog1: TOpenDialog;
    SaveDialog1: TSaveDialog;
    PopupMenu1: TPopupMenu;
    Item1A: TMenuItem;                   // connect
    item1B: TMenuItem;                   // disconnect
    Item2A: TMenuItem;                   // LOG to file
    Item2B: TMenuItem;                   // STOP logging
    Item3A: TMenuItem;                   // paste
    Item3A1: TMenuItem;                  //     from clipboard
    Item3A2: TMenuItem;                  //     from text file
    Item3B: TMenuItem;                   // CANCEL paste
    Item4: TMenuItem;                    // font size
    Item4A: TMenuItem;                   //     9pt
    Item4B: TMenuItem;                   //     12pt
    Item4C: TMenuItem;                   //     14pt
    Item5: TMenuItem;                    // font colour
    Item5A: TMenuItem;                   //     red
    Item5B: TMenuItem;                   //     green
    Item5C: TMenuItem;                   //     yellow
    Item5D: TMenuItem;                   //     blue
    Item5E: TMenuItem;                   //     magenta
    Item5F: TMenuItem;                   //     cyan
    Item5G: TMenuItem;                   //     white
    Item6: TMenuItem;                    // clear
    Item6A: TMenuItem;                   //     GFX plane
    Item6B: TMenuItem;                   //     text plane
    Item6C: TMenuItem;                   //     ring buffer
    N1: TMenuItem;
    Item7: TMenuItem;                    // select and copy
    Item8: TMenuItem;                    // command window
    N2: TMenuItem;
    Item9: TMenuItem;                    // (c) Robert Rozee  2017

    procedure FormCreate(Sender: TObject);
    procedure FormClose(Sender: TObject; var Action: TCloseAction);

    procedure Timer1Timer(Sender: TObject);
    procedure Timer2Timer(Sender: TObject);
    procedure Timer3Timer(Sender: TObject);
    procedure Timer4Timer(Sender: TObject);

    procedure FormKeyPress(Sender: TObject; var Key: Char);
    procedure FormKeyUp(Sender: TObject; var Key: Word;
      Shift: TShiftState);
    procedure FormMouseMove(Sender: TObject; Shift: TShiftState; X,
      Y: Integer);
    procedure FormMouseDown(Sender: TObject; Button: TMouseButton;
      Shift: TShiftState; X, Y: Integer);
    procedure PopupMenu1Popup(Sender: TObject);
    procedure Item1Click(Sender: TObject);
    procedure Item2Click(Sender: TObject);
    procedure Item3Click(Sender: TObject);
    procedure Item4Click(Sender: TObject);
    procedure Item6Click(Sender: TObject);
    procedure Item7Click(Sender: TObject);
    procedure Item8Click(Sender: TObject);
    procedure Item5Click(Sender: TObject);
  private
    { Private declarations }
    procedure CMDialogKey(var Msg: TCMDialogKey); message CM_DIALOGKEY;
    procedure WMSysCommand(var Msg: TWMSysCommand); message WM_SYSCOMMAND;
  public
    { Public declarations }
    procedure WMKeyDown(var Msg: TWMKeyDown); message WM_KEYDOWN;
  end;

var
  Form1: TForm1;

implementation

uses Unit2;

{$R *.DFM}

procedure TForm1.CMDialogKey(var Msg: TCMDialogKey);   // allows tab key to be used
begin
  if Msg.CharCode<>VK_TAB then inherited
end;

procedure TForm1.WMSysCommand(var Msg: TWMSysCommand); // disable hotkey passthrough
begin                                                  // (alt-keys producing error beep)
  if Msg.CmdType<>SC_KEYMENU then inherited
end;

(*
var  hRead_local:THandle;
    hWrite_local:THandle;
     hRead_remote:THandle;
    hWrite_remote:THandle;
*)

var LogFile:text;


var CSR:TLabel;                          // used as a shortcut to cursor object
    SCR:TCanvas;                         // used as a shortcut to text screen
    GFX:TCanvas;                         // used as a shortcut to graphics screen

const SoftwareKey='\Software\NZ made\GFXterm';

const FGdefault:integer=7;               // default text FG colour
      BGdefault:integer=0;               // default text BG colour
       FGcolour:integer=7;               // text foreground colour
       BGcolour:integer=0;               // text background colour
       TxtStyle:TFontStyles=[];          // initial style (underline, bold, etc)
        DimText:boolean=false;           // low intensity flag
        InvText:boolean=false;           // inverse video flag

           Xpos:integer=1;               // initial cursor column
           Ypos:integer=1;               // initial cursor row

          lastK:char=#$00;               // last ascii key pressed
          lastC:char=#$00;               // last ascii character printed

         mouseX:integer=-1;              // last mouse X position
         mouseY:integer=-1;              // last mouse Y position

            CLC:integer=0;               // counter of lines added to command log

            TC1:DWORD=0;                 // timestamp of last serial port data read
            TC2:DWORD=0;                 // timestamp of last keyboard event
            TC3:DWORD=0;                 // timestamp of last VT command decoded

             PB:record
               str:string;               // paste buffer (characters to paste)
               len:integer;              // length of buffer
               idx:integer               // index into buffer
             end=(str:''; len:0; idx:0);

          RB:record
               data:array [0..65535] of char;
               head:integer;
               tail:integer
             end=(head:0; tail:0);

const BreakCounter:integer=-1;           // used to time sending a break
         CONNECTED:boolean=false;        // true if serial port connected
         LOGTOFILE:boolean=false;        // true if logging text to a file
              KILL:boolean=false;        // kill signal to application loop
              DEAD:boolean=false;        // response from application loop

const ROWS=24;
      COLS=80;

var TextStore:array [1..ROWS] of array [1..COLS] of char;

type str255=string[255];


const CPC:array [0..15] of TColor=(clBlack, clNavy, clGreen, clTeal,
                                 clMaroon, clPurple, clOlive, clSilver,
                                 clGray, clBlue, clLime, clAqua,
                                 clRed, clFuchsia, clYellow, clWhite);
{   (dim)           (bright)
0 = Black 	8 = Gray
1 = Blue 	9 = Light Blue
2 = Green 	A = Light Green
3 = Aqua 	B = Light Aqua     (Cyan)
4 = Red 	C = Light Red
5 = Purple 	D = Light Purple   (Magenta)
6 = Yellow 	E = Light Yellow
7 = White 	F = Bright White
}
const CVT:array [0..15] of TColor=(clBlack, clMaroon, clGreen, clOlive,
                                   clNavy, clPurple, clTeal, clSilver,
                                   clGray, clRed, clLime, clYellow,
                                   clBlue, clFuchsia, clAqua, clWhite);
{       (0-7 = dim, 8-15 = bright)
30	Black
31	Red
32	Green
33	Yellow
34	Blue                  TColor($FF8080) = bright blue, as per teraterm
35	Magenta   (Purple)
36	Cyan      (Aqua)
37	White
}

// returns how much time has elapsed since GetTickCount was assigned to counter
function timesince(counter:DWORD):DWORD;
var I:int64;
begin
  I:=GetTickCount-counter;
  if I<0 then I:=I+$100000000;
  result:=I
end;


////////////////////////////////////////////////////////////////////////////////
// the following routines are the text plane primatives. this plane sits
// behind the graphics plane. the following commands are provided:
// putch (x, y, character)              - puts a single character at (x,y)
// gotoxy (x, y)                        - moves cursor to (x,y)
// clear (x1, y1, x2, y2)               - clears a rectangular area
// scroll (x1, y1, x2, y2, direction)   - direction can be -1 (down) or +1 (up)
// emit (ch)                            - print (ch) at cursor, update cursor
////////////////////////////////////////////////////////////////////////////////

procedure putch(X, Y:integer; ch:char);
var mask:byte;
begin
  if DimText then mask:=$0
             else mask:=$8;
  if InvText then begin
                    SCR.Font.Color:=CVT[BGcolour];
                    SCR.Brush.Color:=CVT[FGColour or mask]
                  end
             else begin
                    SCR.Font.Color:=CVT[FGcolour or mask];
                    SCR.Brush.Color:=CVT[BGColour]
                  end;
  SCR.Font.Style:=TxtStyle;
  SCR.TextOut(CSR.Width*(X-1), CSR.Height*(Y-1), ch);

  if (X in [1..COLS]) and (Y in [1..ROWS]) then TextStore[Y, X]:=ch;
  lastC:=ch
end;


procedure gotoxy(X, Y:integer);        // -1 indicate no change
begin
  if X<>-1 then
  begin
    Xpos:=X;               // Xpos = 1 .. ROWS
    if Xpos<1 then Xpos:=1;
    if Xpos>COLS then Xpos:=COLS
  end;

  if Y<>-1 then
  begin
    Ypos:=Y;               // Ypos = 1 .. COLS
    if Ypos<1 then Ypos:=1;
    if Ypos>ROWS then Ypos:=ROWS
  end;

  CSR.Tag:=1                           // flag cursor position as moved
end;


procedure clear(X1, Y1, X2, Y2:integer);
var X, Y:integer;
begin
  if X2<X1 then begin X:=X1; X1:=X2; X2:=X end;
  if Y2<Y1 then begin Y:=Y1; Y1:=Y2; Y2:=Y end;

  SCR.Brush.Color:=CVT[BGColour];
  SCR.FillRect(Rect(CSR.Width*(X1-1), CSR.Height*(Y1-1),
                    CSR.Width*(X2),   CSR.Height*(Y2)  ));

  for Y:=Y1 to Y2 do FillChar(TextStore[Y, X1], X2-X1+1, ' ')
end;


procedure scroll(X1, Y1, X2, Y2, direction:integer);   // +1 = scroll up, -1 = scroll down
var R1, R2:TRect;
      X, Y:integer;
begin
  if X2<X1 then begin X:=X1; X1:=X2; X2:=X end;
  if Y2<Y1 then begin Y:=Y1; Y1:=Y2; Y2:=Y end;

  R1:=Rect(CSR.Width*(X1-1), CSR.Height*(Y1-1),        // upper rectangle
           CSR.Width*(X2)  , CSR.Height*(Y2-1));
  R2:=Rect(CSR.Width*(X1-1), CSR.Height*(Y1  ),        // lower rectangle
           CSR.Width*(X2)  , CSR.Height*(Y2)  );

  SCR.Brush.Color:=CVT[BGColour];
  case direction of +1:begin                           // scroll screen upwards by 1 line
                         SCR.CopyRect(R1, SCR, R2);    // destination=R1, canvas, source=R2
                         R2.Top:=R1.Bottom;
               //        inc(R2.Bottom);
               //        inc(R2.Right);
                         SCR.FillRect(R2);
                         for Y:=Y1 to Y2-1 do move(TextStore[Y+1, X1], TextStore[Y, X1], X2-X1+1);
                                              //   source,             destination,      count
                         FillChar(TextStore[Y2, X1], X2-X1+1, ' ')     // blank line at bottom
                       end;
                    -1:begin;                          // scroll screen downwards by 1 line
                         SCR.CopyRect(R2, SCR, R1);    // destination=R2, canvas, source=R1
                         R1.Bottom:=R2.Top;    // -1;
               //        inc(R1.Right);
                         SCR.FillRect(R1);
                         for Y:=Y2 downto Y1+1 do move(TextStore[Y-1, X1], TextStore[Y, X1], X2-X1+1);
                                                  //   source,             destination,      count
                         FillChar(TextStore[Y1, X1], X2-X1+1, ' ')     // blank line at top
                       end
                 else ShowMessage('invalid scroll value (+1,-1 required)')
  end  { of case }
end;


procedure emit(ch:char);
begin
  lastC:=ch;
  if ch<#32 then case ch of #07:windows.beep(440,250);               // bell
                            #08:if Xpos<>1 then dec(Xpos);           // backspace
                            #09:begin
                                  if Xpos>COLS then
                                  begin
                                    Xpos:=1;
                                    inc(Ypos)
                                  end;
                                  repeat                             // tab
                                    putch(Xpos, Ypos, ' ');
                                    inc(Xpos)
                                  until (Xpos-1) mod 8=0
                                end;
                            #10:begin                                // linefeed
                                  inc(Ypos);
                                  if Ypos>ROWS then
                                  begin
                                    scroll(1, 1, COLS, ROWS, +1);
                                    Ypos:=ROWS
                                  end
                                end;
                            #13:Xpos:=1;                             // carriage return
                            #17:;                                    // DC 1
                            #18:;                                    // DC 2
                            #19:;                                    // DC 3
                            #20:;                                    // DC 4
                            #27:;                                    // escape
                 end  { of case }
            else begin
                   if Xpos>COLS then
                   begin
                     Xpos:=1;
                     inc(Ypos)
                   end;

                   if Ypos>ROWS then
                   begin
                     scroll(1, 1, COLS, ROWS, +1);
                     Ypos:=ROWS
                   end;
                   putch(Xpos, Ypos, ch);
                   inc(Xpos)
                 end;

  if ch in [#8,#9,#10,#13,#32..#255] then CSR.Tag:=1;      // flag cursor position as moved

  if LOGTOFILE and (ch in [#8,#9,#10,#13,#32..#255]) then
  try
    Write(Logfile, ch)                                     // write to log file
  except
    try closefile(LogFile) except end;
    LOGTOFILE:=false
  end
end;


////////////////////////////////////////////////////////////////////////////////
// the following routines are the graphic plane primatives. this plane sits
// in front of the text plane. the following commands are provided:
// Gw                              - returns width of graphic area
// Gh                              - returns height of graphic area
// GFXclear ((X1, Y1, X2, Y2)      - erase a rectular area
// GFXlineAB (X1, Y1, X2, Y2)      - draw a line from (x1,y1) to (x2,y2)
// GFXarc (X1, Y1, X2, Y2, A1, A2) - draw arc within a rectangle, between A1 and A2 degrees
// GFXplot (X, Y)                  - plot a single pixel
// GFXink (R, G, B, width)         - set ink colour and pen width
// GFXfill (X, Y)                  - fill an area we have just enclosed
// GFXmoveto (X, Y)                - set starting location
// DFXdrawto (X, Y)                - draw from previous location to (x,y)
// GFXscroll(X1, Y1, X2, Y2,
//           deltaX, deltaY)       - scroll a graphics area
////////////////////////////////////////////////////////////////////////////////

function Gw:integer;
begin
  result:=Form1.Image2.Picture.Graphic.Width
end;


function Gh:integer;
begin
  result:=Form1.Image2.Picture.Graphic.Height
end;


procedure GFXclear(X1, Y1, X2, Y2:integer);
begin
  GFX.Brush.Color:=clBlack;
  GFX.FillRect(Rect(X1, Y1, X2, Y2))   //  excludes RHS and bottom edge
end;


procedure GFXlineAB(X1, Y1, X2, Y2:integer);
begin
  GFX.MoveTo(X1, Y1);
  GFX.LineTo(X2, Y2)                   // draw line, excluding the last point
//GFX.LineTo(X2, Y2)                   // fill in last point ????????????????
end;


procedure GFXarc(X1, Y1, X2, Y2:integer; A1, A2:single);
var X0, Y0, X3, Y3, X4, Y4:integer;
begin
  A1:=(A1*pi/180.0)-(pi/2.0);            // convert A1 to radians, shift origin ccw 1/4 turn
  A2:=(A2*pi/180.0)-(pi/2.0);            // convert A2 to radians, shift origin ccw 1/4 turn

  X0:=(X1 + X2) div 2;                   // locate centre of elipse: X0
  Y0:=(Y1 + Y2) div 2;                   // locate centre of elipse: Y0

  X3:=X0 + trunc(1000.0*Cos(A2));
  Y3:=Y0 + trunc(1000.0*Sin(A2));
  X4:=X0 + trunc(1000.0*Cos(A1));
  Y4:=Y0 + trunc(1000.0*Sin(A1));

  GFX.Arc(X1, Y1, X2, Y2, X3, Y3, X4, Y4)
end;


procedure GFXplot(X, Y:integer);
begin
  GFX.Pixels[X, Y]:=GFX.Pen.Color
end;


procedure GFXink(R, G, B, width:integer);
begin
  R:=min(255, max(R, 0));
  G:=min(255, max(G, 0));
  B:=min(255, max(B, 0));
  GFX.Pen.Color:=(B*$10000) + (G*$100) + R;
  GFX.Pen.Width:=width
end;


procedure GFXfill(X, Y:integer);
begin
  GFX.Brush.Color:=GFX.Pen.Color;                  // select pen colour as fill
  GFX.FloodFill(X, Y, GFX.Pen.Color, fsBorder);    // fill everything enclosed by pen colour
  GFX.Brush.Color:=clBlack                         // go back to a black brush
(*
  GFX.Brush.Color:=clBlack;                        // select black fill colour
  GFX.FloodFill(X, Y, GFX.Pen.Color, fsBorder);    // fill area bounded by pen colour
  GFX.Brush.Color:=GFX.Pen.Color;                  // select pen colour as fill
  GFX.FloodFill(X, Y, clBlack, fsSurface);         // fill everything we previously set to black
  GFX.Brush.Color:=clBlack                         // go back to a black brush
*)
end;


procedure GFXmoveto(X, Y:integer);
begin
  GFX.MoveTo(X, Y)                                 // set starting point of a multi-line
end;


procedure GFXdrawto(X, Y:integer);
begin
  GFX.LineTo(X, Y)                                 // draw line to (x,y)
end;                                               // excluding the last point


procedure GFXscroll_OLD(X1, Y1, X2, Y2, deltaX, deltaY:integer);
var R1, R2, Fh, Fv:TRect;
begin
  if (deltaX=0) and (deltaY=0) then exit;      // nothing to do!

// R1 is the source rectangle, R2 is the destingation rectangle
  if deltaX<0 then begin                       // moving *** left ***
                     R1.Left:=X1-deltaX;       // - source left side
                     R1.Right:=X2;             // - source right side
                     R2.Left:=X1;              // - destination left side
                     R2.Right:=X2+deltaX;      // - destination right side

                     Fv.Left:=R2.Right+1;
                     Fv.Right:=X2+1            // +1 as fillrect skips right side
                   end
              else begin                       // moving *** right ***
                     R1.Left:=X1;              // - source left side
                     R1.Right:=X2-deltaX;      // - source right side
                     R2.Left:=X1+deltaX;       // - destination left side
                     R2.Right:=X2;             // - destination right side

                     Fv.Left:=X1;
                     Fv.Right:=R2.Left         // fillrect skips right side
                   end;
  Fv.Top:=Y1;
  Fv.Bottom:=Y2+1;                             // +1 as fillrect skips bottom edge

  if deltaY<0 then begin                       // moving *** up ***
                     R1.Top:=Y1-deltaY;        // - source top side
                     R1.Bottom:=Y2;            // - source bottom side
                     R2.Top:=Y1;               // - destination top side
                     R2.Bottom:=Y2+deltaY;     // - destination bottom side

                     Fh.Top:=R2.Bottom+1;
                     Fh.Bottom:=Y2+1           // +1 as fillrect skips bottom edge
                   end
              else begin                       // moving *** down  ***
                     R1.Top:=Y1;               // - source top side
                     R1.Bottom:=Y2-deltaY;     // - source bottom side
                     R2.Top:=Y1+deltaY;        // - destination top side
                     R2.Bottom:=Y2;            // - destination bottom side

                     Fh.Top:=Y1;
                     Fh.Bottom:=R2.Top         // fillrect skips bottom edge
                   end;
  Fh.Left:=X1;
  Fh.Right:=Y2+1;                              // +1 as fillrect skips right side

  GFX.CopyRect(R2, GFX, R1);    // destination=R2, canvas, source=R1

  GFX.Brush.Color:=clBlack;
  if deltaX<>0 then GFX.FillRect(Fv);
  if deltaY<>0 then GFX.FillRect(Fh)
end;

////////////////////////////////////////////////////////////////////////////////
// assumes CopyRect and FillRect both skip right column and bottom row        //
////////////////////////////////////////////////////////////////////////////////
procedure GFXscroll(X1, Y1, X2, Y2, deltaX, deltaY:integer);
var R1, R2, Fh, Fv:TRect;
begin
  if (deltaX=0) and (deltaY=0) then exit;      // nothing to do!
  inc(X2);
  inc(Y2);

// R1 is the source rectangle, R2 is the destingation rectangle
  if deltaX<0 then begin                       // moving *** left ***
                     R1.Left:=X1-deltaX;       // - source left side
                     R1.Right:=X2;             // - source right side
                     R2.Left:=X1;              // - destination left side
                     R2.Right:=X2+deltaX;      // - destination right side

                     Fv.Left:=R2.Right;
                     Fv.Right:=X2
                   end
              else begin                       // moving *** right ***
                     R1.Left:=X1;              // - source left side
                     R1.Right:=X2-deltaX;      // - source right side
                     R2.Left:=X1+deltaX;       // - destination left side
                     R2.Right:=X2;             // - destination right side

                     Fv.Left:=X1;
                     Fv.Right:=R2.Left         // fillrect skips right side
                   end;
  Fv.Top:=Y1;
  Fv.Bottom:=Y2;                               // +1 as fillrect skips bottom edge

  if deltaY<0 then begin                       // moving *** up ***
                     R1.Top:=Y1-deltaY;        // - source top side
                     R1.Bottom:=Y2;            // - source bottom side
                     R2.Top:=Y1;               // - destination top side
                     R2.Bottom:=Y2+deltaY;     // - destination bottom side

                     Fh.Top:=R2.Bottom;
                     Fh.Bottom:=Y2             // +1 as fillrect skips bottom edge
                   end
              else begin                       // moving *** down  ***
                     R1.Top:=Y1;               // - source top side
                     R1.Bottom:=Y2-deltaY;     // - source bottom side
                     R2.Top:=Y1+deltaY;        // - destination top side
                     R2.Bottom:=Y2;            // - destination bottom side

                     Fh.Top:=Y1;
                     Fh.Bottom:=R2.Top         // fillrect skips bottom edge
                   end;
  Fh.Left:=X1;
  Fh.Right:=Y2;                                // +1 as fillrect skips right side

  GFX.CopyRect(R2, GFX, R1);    // destination=R2, canvas, source=R1

  GFX.Brush.Color:=clBlack;
  if deltaX<>0 then GFX.FillRect(Fv);
  if deltaY<>0 then GFX.FillRect(Fh)
end;

////////////////////////////////////////////////////////////////////////////////
// the follow are the communications (serial) port routines:
// SetupCommPort (CommPortName, Config:string)
// ReadComm (var S:string):boolean
// WriteComm (S:string);
// CommBytesWaiting:cardinal;
////////////////////////////////////////////////////////////////////////////////

var CommFile:THandle;


procedure SetupCommPort(CommPortName, Config:str255);
var DeviceName:array [0..80] of char;
           DCB:TDCB;
       CommTOs:TCommTimeouts;
          proc:str255;
begin
  try
    CommPortName:=copy(CommPortName,1,80);             // 80 characters max
    StrPCopy(DeviceName, '\\.\'+CommPortName);
    proc:='CreateFile';
    CommFile:=CreateFile(DeviceName,
                         GENERIC_READ or GENERIC_WRITE,
                         0, Nil,
                         OPEN_EXISTING,
                         FILE_ATTRIBUTE_NORMAL, 0);
    if (CommFile=INVALID_HANDLE_VALUE) then
    begin
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end;
    proc:='SetupComm';
    if not SetupComm(CommFile, 32768, 1024) then
    begin
      try CloseHandle(CommFile) except end;
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end;
    proc:='GetCommState';
    if not GetCommState(CommFile, DCB) then
    begin
      try CloseHandle(CommFile) except end;
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end;
//    Config:='baud='+IntToStr(38400)+' parity=n data=8 stop=1 xon=off'#0;   // port parameters
    Config:=Config+#0;                                                       // xon={on|off}
    proc:='BuildCommDCB';
    if not BuildCommDCB(@Config[1], DCB) then
    begin
      try CloseHandle(CommFile) except end;
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end;
    proc:='SetCommState';
    if not SetCommState(CommFile, DCB) then
    begin
      try CloseHandle(CommFile) except end;
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end;
    with CommTOs do
    begin
      ReadIntervalTimeout:=1;                  // 10;  // 0
      ReadTotalTimeoutMultiplier:=0;           // 0
      ReadTotalTimeoutConstant:=1;             // 10;  // 300
      WriteTotalTimeoutMultiplier:=0;          // 0
      WriteTotalTimeoutConstant:=0;            // 10   // 300
    end;
    proc:='SetCommTimeouts';
    if not SetCommTimeouts(CommFile, CommTOs) then
    begin
      try CloseHandle(CommFile) except end;
      ShowMessage('Serial I/O error: '+proc+' failed');
      exit
    end
  except
    try CloseHandle(CommFile); except end;
    ShowMessage('Serial I/O error: '+proc+' exception');
    exit
  end;
  CONNECTED:=true
end;


function ReadComm(var S:str255):boolean;
const lock:boolean=false;
var get, got, error, reason:DWORD;
                   CommStat:TComStat;
                     buffer:array [1..250] of char;
                       flag:byte;
begin
  result:=false;
  if lock or not CONNECTED then exit;
  lock:=true;
  flag:=0;                                             // to keep compiler happy!
//inc(LC1);
  try
    flag:=1;
    if not ClearCommError(CommFile, reason, @CommStat) then
    begin
      error:=GetLastError;
      try CloseHandle(CommFile); except end;
      CONNECTED:=false;
      ShowMessage('Serial I/O error: ClearCommError failed'+#13+
                  '('+IntToStr(error)+')  '+SysErrorMessage(error));
      lock:=false;
      exit
    end;
    get:=min(CommStat.cbInQue, sizeof(buffer));        // number of characters waiting
//  get:=sizeof(buffer);

    flag:=2;
    if get=0 then got:=0 else                          // skip ReadFile if nothing waiting
    if not ReadFile(CommFile, buffer, get, got, nil) then
    begin
      error:=GetLastError;
      try CloseHandle(CommFile); except end;
      CONNECTED:=false;
      ShowMessage('Serial I/O error: ReadFile failed'+#13+
                  '('+IntToStr(error)+')  '+SysErrorMessage(error));
      lock:=false;
      exit
    end
  except
    try CloseHandle(CommFile); except end;
    CONNECTED:=false;
    if flag=1 then ShowMessage('Serial I/O error: ClearCommError exception')
              else ShowMessage('Serial I/O error: ReadFile exception');
    lock:=false;
    exit
  end;

  S:=StringOfChar(#0, got);
  move(buffer[1], S[1], got);
//S:=copy(buffer, 0, got);

  result:=(got<>0);
  lock:=false
end;


function WriteComm(S:str255):boolean;
const lock:boolean=false;
var put, error:DWORD;
begin
  result:=false;
  if lock or not CONNECTED then exit;
  lock:=true;
//inc(LC2);
  try
    if not WriteFile(CommFile, S[1], length(S), put, nil) then
    begin
      error:=GetLastError;
      try CloseHandle(CommFile); except end;
      CONNECTED:=false;
      ShowMessage('Serial I/O error: WriteFile failed'+#13+
                  '('+IntToStr(error)+')  '+SysErrorMessage(error));
      lock:=false;
      exit
    end
  except
    try CloseHandle(CommFile); except end;
    CONNECTED:=false;
    ShowMessage('Serial I/O error: WriteFile exception');
    lock:=false;
    exit
  end;

  result:=true;
  lock:=false
end;


function CommBytesWaiting:cardinal;
var CommStat:TComStat;
      Errors:cardinal;
begin
  if ClearCommError(CommFile, Errors, @CommStat) then result:=CommStat.cbInQue
                                                 else result:=0
end;


////////////////////////////////////////////////////////////////////////////////
// the following is a VT100 command processing engine. it takes characters one
// at a time via the parameter 'ch'. normally these are simply passed out
// unaltered as a one character long return string to then be passed onto
// emit(ch).
//
// however, when a VT100 command sequence is detected characters are accumulated
// in the internal string 'VTline'. once matched to a valid VT100 command, the
// command is enacted. while if the string accumulated is detected as invalid,
// the whole string is returned.
////////////////////////////////////////////////////////////////////////////////

// const IgnoreCount:integer=0;

function VT100engine(ch:char):str255;
const VTflag:boolean=false;         // internally held flag
      VTline:str255='';             // internally held string
       Xsave:integer=-1;            //  )
       Ysave:integer=-1;            //  )
      FGsave:integer=-1;            //  ) saved values for
      BGsave:integer=-1;            //  ) cursor location
      TSsave:TFontStyles=[];        //  ) and attributes
      DTsave:boolean=false;         //  )
      ITsave:boolean=false;         //  )
          TM:integer=-maxint;
          BM:integer=+maxint;
var n, v, h, I, J:integer;
                S:str255;

  function PC(S:str255):integer;       // returns count of semicolon delimited parameters
  var I, dc, sc:integer;               // or -1 if any invalid characters are found
  begin
    if length(S)=0 then PC:=0 else     // empty string -> no parameters
    begin
      dc:=0;                           // number of digits
      sc:=0;                           // number of semicolon delimiters
      for I:=1 to length(S) do
      begin
        if S[I] in ['0'..'9'] then inc(dc);
        if S[I]=';' then inc(sc);
      end;
      if (dc+sc)<>length(S) then PC:=-1        // invalid characters found
                            else PC:=sc+1      // number of parameters in S
    end
  end;      // ( 1;2 = 2 parameters, 1;;2 = 3 parameters, ;;; = 4 parameters

  function Pn(S:str255; n:integer):integer;    // returns parameter n as an integer
  var I:integer;
  begin
    while (n>1) and (length(S)<>0) do
    begin
      if pos(';', S)=1 then dec(n);
      delete(S, 1, 1)
    end;
    if (length(S)=0) or (S[1]=';') then Pn:=0 else
    begin
      I:=pos(';', S);
      if I<>0 then S:=copy(S, 1, I-1);
      if length(S)=0 then Pn:=0
                     else try
                            Pn:=StrToInt(S)
                          except
                            Pn:=0              // should never arrive here
                          end
    end
  end;

  function OK(S:str255; var n:integer):boolean;        // returns true if a valid number
  begin                                                // (with number placed in n)
    OK:=true;
    try
      if length(S)=0 then n:=0
                     else n:=StrToInt(S)
    except
      OK:=false
    end
  end;

  procedure fail;              // failed to decode string
  begin
    if Form2.Visible then with Form2.RichEdit1 do
    begin
      inc(CLC);
      Lines.BeginUpdate;
      if CLC>999 then Lines.Delete(0);
      Lines.Add(Format('%.6d  ',[CLC mod 1000000])+
                'FAIL: <ESC>'+copy(VTline,2,length(VTline)-1));
      SendMessage(Handle, WM_VSCROLL, SB_BOTTOM, 0);
      Lines.EndUpdate
    end;

    result:=VTline;            // return complete string
    VTline:='';
    VTflag:=false              // drop out of VT100 mode
  end;

  procedure pass;              // sucessfully processed string
  begin
    if Form2.Visible then with Form2.RichEdit1 do
    begin
      inc(CLC);
      Lines.BeginUpdate;
      if CLC>999 then Lines.Delete(0);
      Lines.Add(Format('%.6d  ',[CLC mod 1000000])+
                'PASS: <ESC>'+copy(VTline,2,length(VTline)-1));
      SendMessage(Handle, WM_VSCROLL, SB_BOTTOM, 0);
      Lines.EndUpdate
    end;

    VTline:='';
    VTflag:=false;             // drop out of VT100 mode
    TC3:=GetTickCount
  end;

begin
  if ch=#27 then VTflag:=true;
  if not VTflag then result:=ch else
  begin
    result:='';
    VTline:=VTline+ch;
    if length(VTline)=1 then exit;

    if pos(#27+'[', VTline)=1 then             // process a complete <esc>[ string
    begin
      if length(VTline)=2 then exit;           // still building command string
      if ch in ['0'..'9',';'] then exit;       // still building command string

      S:=copy(VTline, 3, length(VTline)-3);    // extract parameter segment

      case ch of 'A':begin                     // move cursor up n lines
                       if not OK(S, n) then fail else
                       begin
//                       ShowMessage('UP '+IntToStr(n));
                         if n=0 then n:=1;
                         gotoxy(-1, Ypos-n);
                         pass
                       end
                     end;
                 'B':begin                     // move cursor down n lines
                       if not OK(S, n) then fail else
                       begin
//                       ShowMessage('DOWN '+IntToStr(n));
                         if n=0 then n:=1;
                         gotoxy(-1, Ypos+n);
                         pass
                       end
                     end;
                 'C':begin                     // move cursor right n columns
                       if not OK(S, n) then fail else
                       begin
//                       ShowMessage('RIGHT '+IntToStr(n));
                         if n=0 then n:=1;
                         gotoxy(Xpos+n, -1);
                         pass
                       end
                     end;
                 'D':begin                     // move cursor left n columns
                       if not OK(S, n) then fail else
                       begin
//                       ShowMessage('LEFT '+IntToStr(n));
                         if n=0 then n:=1;
                         gotoxy(Xpos-n, -1);
                         pass
                       end
                     end;
             'H','f':begin                     // position cursor
                       n:=PC(S);
//                     showmessage('position cursor |'+S+'|'+VTline+'| '+IntToStr(n));
                       case n of 0:begin
                                     gotoxy(1, 1);
//                                   ShowMessage('top left');
                                     pass
                                   end;
                                 1:begin
                                     v:=Pn(S,1);
                                     if v=0 then v:=1;
                                     gotoxy(1, v);
                                     pass
                                   end;
                                 2:begin
                                     v:=Pn(S,1);
                                     h:=Pn(S,2);
                                     if v=0 then v:=1;
                                     if h=0 then h:=1;
                                     gotoxy(h, v);
//                                   showmessage('row = '+IntToStr(v)+'  col = '+IntToStr(h));
                                     pass
                                   end
                              else fail
                       end  { of case }
                     end;
                 'J':if not OK(S, n) then fail else    // clear screen above/below cursor
                     case n of 0:begin                 // clear cursor to end of screen
                                   clear(Xpos, Ypos, COLS, Ypos);
                                   if Ypos<>ROWS then clear(1, Ypos, COLS, ROWS);
                                   pass
                                 end;
                               1:begin                 // clear start of screen to cursor
                                   if Ypos<>1 then clear(1, 1, COLS, Ypos-1);
                                   clear(1, Ypos, Xpos, Ypos);
                                   pass
                                 end;
                               2:begin                 // clear whole screen
                                   clear(1, 1, COLS, ROWS);
//                                 gotoxy(1, 1);      // ANSI std *may* specify home cursor
                                   pass
                                 end
                            else fail
                     end;  { of case }
                 'K':if not OK(S, n) then fail else    // clear line to left/right of cursor
                     case n of 0:begin                 // clear cursor to EOL
                                   clear(Xpos, Ypos, COLS, Ypos);
                                   pass
                                 end;
                               1:begin                 // clear SOL to cursor
                                   clear(1, Ypos, Xpos, Ypos);
                                   pass
                                 end;
                               2:begin                 // clear whole line
                                   clear(1, Ypos, COLS, Ypos);
                                   pass
                                 end
                                else fail
                     end;  { of case }
//                   if IgnoreCount<>0 then begin dec(IgnoreCount); pass end else 'm':
                 'm':begin
                       if S='' then S:='0';
                       n:=PC(S);
//                     ShowMessage(IntToStr(n)+'  |'+S+'|');
                       if n=-1 then fail else
                       begin
                         for I:=1 to n do
                         begin
                           J:= Pn(S,I);
                           case J of 0:begin
//                                       ShowMessage('reset colours and attributes');
                                         DimText:=false;
                                         InvText:=false;
                                         FGcolour:=FGdefault;
                                         BGcolour:=BGdefault;
                                         TxtStyle:=[]
                                       end;
                                     1:TxtStyle:= TxtStyle+[fsBold];       // bold
                                     2:DimText:=true;                      // dim FGcolour
                                     4:TxtStyle:=TxtStyle+[fsUnderline];   // underline
                                     5:begin end;                          // (not supported)
                                     7:InvText:=true;                      // reverse video
                                30..37:FGcolour:=J-30;
                                    39:FGcolour:=FGdefault;
                                40..47:BGcolour:=J-40;
                                    49:BGcolour:=BGdefault
                           end  { of case }
                         end;
                         pass
                       end
                     end;
                 'n':if not OK(S, n) then fail else
                     if n<>6 then fail else
                     begin
//                     ShowMessage('cursor position request');
                       WriteComm(#27+'['+IntToStr(Ypos)+';'+IntToStr(Xpos)+'R');
                       pass
                     end;
                 'r':begin               // only works with ESC E, D, M
                       n:=PC(S);
//                     ShowMessage('set scroll window |'+S+'|'+VTline+'| '+IntToStr(n));
                       case n of 0:begin
                                     TM:=-maxint;
                                     BM:=+maxint;
//                                   gotoxy(1, 1);             // home cursor
//                                   ShowMessage('zero parameters');
                                     pass
                                   end;
                                 2:begin
                                     I:=Pn(S,1);
                                     J:=Pn(S,2);
                                     if (I<J) and (I>=1)    then TM:=I
                                                            else TM:=-maxint;
                                     if (I<J) and (J<=ROWS) then BM:=J
                                                            else BM:=+maxint;
//                                   gotoxy(1, 1);             // home cursor
//                                   ShowMessage('top margin = '+IntToStr(I)+
//                                          '  bottom margin = '+IntToStr(J));
                                     pass
                                   end
                              else fail
                       end  { of case }
                     end
              else   fail
      end  { of case }
    end
    else

    if pos(#27, VTline)=1 then                 // process a complete <esc> string
    begin
      if length(VTline)=1 then exit;           // still building command string
      if ch='[' then exit;                     // still building command string

//    S:=copy(VTline, 2, length(VTline)-2);    // extract parameter segment

      case ch of '7':begin                     // save cursor position and attributes
                       Xsave:=Xpos;
                       Ysave:=Ypos;
                       FGsave:=FGcolour;
                       BGsave:=BGcolour;
                       TSsave:=TxtStyle;
                       DTsave:=DimText;
                       ITsave:=InvText;
                       pass
                     end;
                 '8':begin                     // restore cursor position and attributes
                       if (Xsave>0) and (Ysave>0) then
                       begin
                         gotoxy(Xsave,Ysave);
                         FGcolour:=FGsave;
                         BGcolour:=BGsave;
                         TxtStyle:=TSsave;
                         DimText:=DTsave;
                         InvText:=ITsave;
                         Xsave:=-1;
                         Ysave:=-1
                       end;
                       pass
                     end;
                 'E':begin                     // same as ESC D (below) plus a CR
//                     ShowMessage('esc-E');
                       if Ypos<min(COLS, BM) then inc(Ypos) else
                       begin
                         scroll(1, max(1, TM), COLS, min(ROWS, BM), +1);
                       end;
                       Xpos:=1;
                       pass
                     end;
                 'D':begin                     // scroll up if cursor is at bottom of window
//                     ShowMessage('esc-D');
                       if Ypos<min(COLS, BM) then inc(Ypos) else
                       begin
                         scroll(1, max(1, TM), COLS, min(ROWS, BM), +1);
                       end;
                       pass
                     end;
                 'M':begin                     // scroll down if cursor is at top of window
//                     ShowMessage('esc-M');
                       if Ypos>max(1, TM) then dec(Ypos) else
                       begin
                         scroll(1, max(1, TM), COLS, min(ROWS, BM), -1);
                       end;
                       pass
                     end
              else   fail
      end;  { of case }
    end else fail
  end
end;


////////////////////////////////////////////////////////////////////////////////
// the following is a GFX command processing engine. it takes characters one
// at a time via the parameter 'ch'. normally these are simply passed out
// unaltered as a one character long return string to then be passed onto
// emit(ch).
//
// however, when a GFX command sequence is detected characters are accumulated
// in the internal string 'GFXline'. once matched to a valid GFX command, the
// command is enacted. while if the string accumulated is detected as invalid,
// the whole string is returned.
////////////////////////////////////////////////////////////////////////////////

function GFXengine(ch:char):str255;
const GFXflag:boolean=false;        // internally held flag
      GFXline:str255='';            // internally held string
      GFXpass:boolean=false;        // true if we've just succeeded in decoding
var params:array[1..16] of integer;
    CMD, S, PSn:str255;
        I, J, n:integer;

  procedure fail;             // failed to decode string
  begin
    if Form2.Visible then with Form2.RichEdit1 do
    begin
      inc(CLC);
      Lines.BeginUpdate;
      if CLC>999 then Lines.Delete(0);
      Lines.Add(Format('%.6d  ',[CLC mod 1000000])+
                'FAIL: <DLE>'+copy(GFXline,2,length(GFXline)-2));
      SendMessage(Handle, WM_VSCROLL, SB_BOTTOM, 0);
      Lines.EndUpdate
    end;

    result:=GFXline;          // return complete string
    GFXline:='';
    GFXflag:=false            // drop out of GFX mode
  end;

  procedure pass;             // sucessfully processed string
  begin
    if Form2.Visible then with Form2.RichEdit1 do
    begin
      inc(CLC);
      Lines.BeginUpdate;
      if CLC>999 then Lines.Delete(0);
      Lines.Add(Format('%.6d  ',[CLC mod 1000000])+
                'PASS: <DLE>'+copy(GFXline,2,length(GFXline)-2));
      SendMessage(Handle, WM_VSCROLL, SB_BOTTOM, 0);
      Lines.EndUpdate
    end;

    GFXline:='';
    GFXflag:=false;           // drop out of GFX mode
    GFXpass:=true
  end;

begin
  if GFXpass and (ch=#10) then       // supress trailing LF for BASIC compatibility
  begin
    GFXpass:=false;
    result:='';
    exit
  end;

  if ch=#16 then GFXflag:=true;
  if not GFXflag then result:=ch else
  begin
    result:='';
    GFXline:=GFXline+ch;

    if (GFXline[1]=#16) and (GFXline[length(GFXline)]=#13) then
    begin                          // process a complete <dle> string
//  format is:  <DLE> Command [,|<TAB>|<SPC>]  Param1 [,|<TAB><SPC>]  Param2... <cr><lf>

      S:=GFXline;

// the below FOR and WHILE loops perform the following:
// - convert every control character, space, comma and semicolon into a TAB
// - convert all consecutive groups of TABs into a single TAB
// - remove any leading TABs from the beginning
// - remove any trailing TABs from the end
      for I:=1 to length(S) do if S[I] in [#00..#32,',',';'] then S[I]:=#09;

      I:=pos(#09#09,S);
      while I<>0 do                // convert pairs of TABs into single TABs
      begin
        delete(S, I, 1);
        I:=pos(#09#09,S)
      end;

      while (length(S)<>0) and (S[1]=#09) do
          delete(S, 1, 1);                     // remove leading TABs
      while (length(S)<>0) and (S[length(S)]=#09) do
          delete(S, length(S), 1);             // remove trailing TABs

      if length(S)=0 then fail;
      if not GFXflag then exit;

// next the command is peeled off the start of the string, and converted to upper case
(*
      I:=1;
      while (I<=length(S)) and (S[I] in ['a'..'z', 'A'..'Z','?']) do inc(I);  // first non-letter
      if I>length(S) then I:=0                                                // none found
                     else if S[I]<>#9 then insert(#9,S,I);   // hack to put tab after command
*)
      I:=pos(#09, S);                // old version, fails with negative # following command

      if I<>0 then begin
                     CMD:=UpperCase(copy(S, 1, I-1));
                     delete(S, 1, I)
                   end
              else begin
                     CMD:=S;
                     S:=''
                   end;

// now we (1) abbreviate/shorten the command to a single letter, and,
//        (2) decide how many parameters we are expecting to see
      if length(CMD)<>1 then
      if CMD='CLEAR'  then CMD:='C' else
      if CMD='INK'    then CMD:='I' else
      if CMD='LINE'   then CMD:='L' else
      if CMD='PLOT'   then CMD:='P' else
      if CMD='ARC'    then CMD:='A' else
      if CMD='FILL'   then CMD:='F' else
      if CMD='MOVETO' then CMD:='M' else
      if CMD='DRAWTO' then CMD:='D' else
      if CMD='SCROLL' then CMD:='S'
                      else fail;               // failed to identify a long command
      if not GFXflag then exit;

      n:=0;                                    // suppress compiler warning
      case CMD[1] of 'P', 'F', 'M', 'D':n:=2;
                          'C', 'I', 'L':n:=4;
                               'A', 'S':n:=6;
                                    '?':n:=0
                  else                  fail   // failed to identify a short command
      end;  { of case }
      if not GFXflag then exit;

// the next thing to do is to separate out the parameters
      for I:=1 to n do
      begin
        J:=pos(#09, S);
        if J<>0 then begin
                       PSn:=copy(S, 1, J-1);
                       delete(S, 1, J)
                     end
                else begin
                       PSn:=S;
                       S:=''
                     end;

        try
          params[I]:=trunc(StrToFloat(PSn))
        except
          fail;
          exit
        end
      end;

      case CMD[1] of 'C':GFXclear(params[1], params[2], params[3], params[4]);
                     'I':GFXink(params[1], params[2], params[3], params[4]);
                     'L':GFXlineAB(params[1], params[2], params[3], params[4]);
                     'P':GFXplot(params[1], params[2]);
                     'A':GFXarc(params[1], params[2], params[3], params[4], params[5], params[6]);
                     'F':GFXfill(params[1], params[2]);
                     'M':GFXmoveto(params[1], params[2]);
                     'D':GFXdrawto(params[1], params[2]);
                     'S':GFXscroll(params[1], params[2], params[3], params[4], params[5], params[6]);
                     '?':WriteComm(IntToStr(Gw)+', '+IntToStr(Gh)+#13)
                  else   fail
      end;  { of case }
      if GFXflag then pass
    end  { end of processing a valid GFX command string }
  end
end;


////////////////////////////////////////////////////////////////////////////////
// Timer event handlers (x3)
// Timer1: 500mS, flash cursor
// Timer2: 20mS, update status bar and cursor location
// Timer3: 20mS, process incoming and outgoing serial data
////////////////////////////////////////////////////////////////////////////////

procedure TForm1.Timer1Timer(Sender: TObject);     // flash cursor (slow)
const flag:boolean=false;                          // 500ms typical = 1/2 sec.
begin
  flag:=not flag;
  CSR.Visible:=flag or (CSR.Tag=1);                // cursor always visible if it has moved
  CSR.Tag:=0;

  if flag then Label8.Color:=clRed
          else Label8.Color:=clLime
end;


procedure TForm1.Timer2Timer(Sender: TObject);     // update status bar and cursor position
var I:int64;                                       // 50ms typical = 1/20 sec.
    S:str255;
    L:integer;
begin
  Label1.Caption:=Format('%.2x', [ord(lastC)]);
  Label2.Caption:=Format('row=%.2d', [Ypos]);
  Label3.Caption:=Format('col=%.2d', [Xpos]);
  Label4.Caption:=Format('key=%.2x', [ord(lastK)]);

  I:=timesince(TC1);
  S:=Format('%.10d', [I]);
  if I<1000  then Label5.Caption:=copy(S,8,3)+'ms' else
  if I<60999 then Label5.Caption:=copy(S,6,2)+'.'+S[8]+'s' else
                  Label5.Caption:='>1min';
  Label5.Visible:=CONNECTED;                   // hide Rx timer if not connected

  I:=timesince(TC2);
  S:=Format('%.10d', [I]);
  if I<1000  then Label6.Caption:=copy(S,8,3)+'ms' else
  if I<60999 then Label6.Caption:=copy(S,6,2)+'.'+S[8]+'s' else
                  Label6.Caption:='>1min';

  L:=RB.head-RB.tail;
  if L<0 then L:=L+sizeof(RB.data);
  L:=min(99, (L*100) div sizeof(RB.data));
  Label7.Caption:=Format('%.2d%%', [L]);       // percentage of (64k) ring buffer used
  if L<10 then Label7.Color:=clBtnFace else
  if L<40 then Label7.Color:=clAqua else
  if L<70 then Label7.Color:=clYellow
          else Label7.Color:=clRed;

  Label8.Visible:=CONNECTED;                   // hide 'online' indicator if not connected
  Label9.Visible:=LOGTOFILE;                   // hide 'logging' indicator if not logging

  Label10.Caption:=Format('[%.6d]', [min(999999, max(PB.len-PB.idx, 0))]);
  if timesince(TC3)>500 then Label10.Color:=clBtnFace
                        else Label10.Color:=clAqua;
  Label10.Visible:=(PB.idx<>0);

  Label11.Caption:='('+IntToStr(mouseX)+','+IntToStr(mouseY)+')';
  Label11.Visible:=(MouseX>=0) and (MouseY>=0);

  CSR.Top :=Image1.Top+ (CSR.Height*(min(max(Ypos, 1), ROWS)-1));
  CSR.Left:=Image1.Left+(CSR.Width* (min(max(Xpos, 1), COLS)-1))
end;


procedure TForm1.Timer3Timer(Sender: TObject);     // serial ticker (fast)
var I1, I2:integer;                                // typical 5ms = 1/200th sec.
    S1, S2:str255;                                 // fallthrough after 20ms
      mark:int64;
        ch:char;
begin
  mark:=GetTickCount;

// *** read from comm port, placing characters into ring buffer ***
  while CONNECTED and ReadComm(S1) do
  begin
    TC1:=GetTickCount;
    for I1:=1 to length(S1) do
    begin
      I2:=(RB.head+1) mod sizeof(RB.data);
      if I2<>RB.tail then
      begin
        RB.data[RB.head]:=S1[I1];
        RB.head:=I2
      end
    end
  end;

// *** for a maximum of 20ms, process characters to screen ***
  while (timesince(mark)<20) and (RB.head<>RB.tail) do
  begin
    ch:=RB.data[RB.tail];
    RB.tail:=(RB.tail+1) mod sizeof(RB.data);

    S1:=VT100engine(ch);                           // first process throught the VT100 engine
    for I1:=1 to length(S1) do
    begin
      S2:=GFXengine(S1[I1]);                       // then process through the GFX engine
      for I2:=1 to length(S2) do
      if S2[I2]=#05 then WriteComm(#06)            // ENQ -> ACK
                    else emit(S2[I2])              // lastly print out the remainder
    end
  end;

// *** handle break (alt-B) key: SetCommBreak, wait 100ms, ClearCommBreak ***
  if BreakCounter<>-1 then                         // BreakCounter is active
  begin
    if BreakCounter=0 then try SetCommBreak(CommFile) except end;
    Inc(BreakCounter, Timer3.Interval);

    if BreakCounter>100 then                       // after 100mS return to idle state
    begin
      try ClearCommBreak(CommFile) except end;
      BreakCounter:=-1
    end
  end
end;


// *** process paste operations, one character per loop through if in editor ***
procedure TForm1.Timer4Timer(Sender: TObject);     // paste ticker (variable)
const ch:char=#00;
var S:string;
    I:integer;
begin
  if (PB.idx<>0) and CONNECTED then
  begin
    if PB.idx=1 then TC3:=GetTickCount;        // force slow paste initially

    if timesince(TC3)>500 then                 // *** fast paste ***
    begin                                      // ==================
      Timer4.Interval:=10;                     // 10ms between checks
      if timesince(TC1)>30 then                // 30ms silence between lines
      begin
        S:='';
        I:=PB.idx;
        repeat
          ch:=PB.str[I];
          S:=S+ch;
          inc(I)
        until (I>PB.len) or (ch=#13);          // grab a line up to and inc CR
        if WriteComm(S) then PB.idx:=I
      end
    end
    else                                       // *** slow paste ***
    begin                                      // ==================
      Timer4.Interval:=3;                      // 3ms between characters
      if not((ch=#13) and (timesince(TC1)<300)) then
      begin                                    // 300ms since last char returned
        ch:=PB.str[PB.idx];
        if WriteComm(ch) then
        begin
          inc(PB.idx);
          if ch=#13 then Timer4.Interval:=300  // 300ms minimum delay after CR
        end
      end
    end;

    if (PB.idx>PB.len) then                    // have finished pasting
    begin
      PB.idx:=0;
      PB.len:=0;
      PB.str:='';
      Timer4.Interval:=200;                    // every 200ms check for paste
      ch:=#00
    end
  end
end;


////////////////////////////////////////////////////////////////////////////////
// Keyboard and Mouse Events
// =========================
////////////////////////////////////////////////////////////////////////////////

// *** handle normal ascii keys ***
procedure TForm1.FormKeyPress(Sender: TObject; var Key: Char);
begin
  if Memo1.Visible then exit;                  // ignore keys if text buffer visible

  lastK:=Key;
  if PB.idx=0 then if CONNECTED then WriteComm(Key)
                                else emit(Key);
  TC2:=GetTickCount
end;


// *** handle special keys that need to be auto-repeating ***
procedure TForm1.WMKeyDown(var Msg:TWMKeyDown);
var S:str255;
begin
  if Memo1.Visible then exit;                  // ignore keys if text buffer visible

  S:='';
  case Msg.CharCode of VK_LEFT:S:=#27+'[D';
                         VK_UP:S:=#27+'[A';
                      VK_RIGHT:S:=#27+'[C';
                       VK_DOWN:S:=#27+'[B';
                      VK_PRIOR:S:=#27+'[5~';
                       VK_NEXT:S:=#27+'[6~';
                       VK_HOME:S:=#27+'[1~';
                        VK_END:S:=#27+'[4~';
                     VK_INSERT:S:=#27+'[2~';
                     VK_DELETE:S:=#127;        // convert delete key to chr(127)
  end;  { of case }

  if CONNECTED and (PB.idx=0) and (length(S)>0) then
  begin
    WriteComm(S);
    TC2:=GetTickCount
  end
end;


// *** handle non-repeating keys upon release ***
procedure TForm1.FormKeyUp(Sender: TObject; var Key: Word; Shift: TShiftState);
var S:str255;
begin
  if Memo1.Visible then                        // ignore keys if text buffer visible
  begin
    if (Key=13) or (Key=27) then               // except hide Memo1 on ESC or CR
    begin
      Form1.PopupMenu:=PopupMenu1;             // re-enable our popup menu

      Memo1.Lines.Clear;
      Memo1.Enabled:=false;                    // probably not needed
      Memo1.Visible:=false;
//    Memo1.SendToBack;                        // probably not needed
//    Form1.SetFocus                           // probably not needed
    end;
    exit
  end;

// ssShift	The Shift key is held down.
// ssAlt	The Alt key is held down.
// ssCtrl	The Ctrl key is held down.

  S:='';                                       // default action: do nothing

// *** shifted and unshifted function keys ***
  if ssShift in Shift
  then case Key of VK_F3:S:=#27+'[25~';        // shifted function keys
                   VK_F4:S:=#27+'[26~';
                   VK_F5:S:=#27+'[28~';
                   VK_F6:S:=#27+'[29~';
                   VK_F7:S:=#27+'[31~';
                   VK_F8:S:=#27+'[32~';
                   VK_F9:S:=#27+'[33~';
                  VK_F10:S:=#27+'[34~'
       end  { of case }
  else case Key of VK_F1:S:=#27+'[11~';        // unshifted function keys
                   VK_F2:S:=#27+'[12~';
                   VK_F3:S:=#27+'[13~';
                   VK_F4:S:=#27+'[14~';
                   VK_F5:S:=#27+'[15~';
                   VK_F6:S:=#27+'[17~';
                   VK_F7:S:=#27+'[18~';
                   VK_F8:S:=#27+'[19~';
                   VK_F9:S:=#27+'[20~';
                  VK_F10:S:=#27+'[21~';
                  VK_F11:S:=#27+'[23~';
                  VK_F12:S:=#27+'[24~'
       end;  { of case }

// *** alt keys (letters and numbers), these are all 'special' commands ***
  if ssAlt in Shift then
  begin
    case Key of ord('B'):if CONNECTED then BreakCounter:=0;    // reset micromite
                ord('C'):GFXclear(0, 0, Gw, Gh);               // clear graphics layer
                ord('D'):clear(1, 1, COLS, ROWS);              // clear text layer
                ord('Z'):begin                                 // cancel paste
                           PB.idx:=0;
                           PB.len:=0;
                           PB.str:=''
                         end;
                ord('A'):RB.head:=RB.tail;                     // clear ring buffer
                ord('M'):Form2.RichEdit1.Lines.Add('++++++++++++++++++++++++');
//              ord('1'):CSR.Font.Size:=9;     // select 9pt text size
//              ord('2'):CSR.Font.Size:=12;    // select 12pt text size
//              ord('3'):CSR.Font.Size:=14     // select 14pt text size
    end;  { of case }

// *** adjust dimensions of everything on screen to match new point size ***
(*  if Key in [ord('1'), ord('2'), ord('3')] then
    begin
      Image1.Width:=CSR.Width*COLS;            // set width
      Image1.Height:=CSR.Height*ROWS;          // set height

      Image2.Width:=Image1.Width;
      Image2.Height:=Image1.Height;

      Memo1.Width:=Image1.Width;
      Memo1.Height:=Image1.Height;
      Memo1.Font:=CSR.Font;
      Memo1.Font.Color:=clBlack;
      Memo1.Font.Style:=[];

      Image1.Picture.Graphic.Width:=Image1.Width;      // set width
      Image1.Picture.Graphic.Height:=Image1.Height;    // set height

      Image2.Picture.Graphic.Width:=Image2.Width;      // set width
      Image2.Picture.Graphic.Height:=Image2.Height;    // set height

      SCR.Font:=Cursor.Font;                   // copy font details from cursor object

      clear(1, 1, COLS, ROWS);                 // clear text layer
      GFXclear(0, 0, Gw, Gh);                  // clear graphics layer
      gotoxy(1,1)                              // home cursor
    end  *)
  end;

//if Key=VK_Delete then S:=#127;               // convert delete key to chr(127)
  if Key=VK_F10 then Key:=0;                   // stop f10 triggering menu

  if CONNECTED and (PB.idx=0) and (length(S)>0) then
  begin
    WriteComm(S);
    TC2:=GetTickCount
  end
end;


procedure TForm1.FormMouseDown(Sender: TObject; Button: TMouseButton;
  Shift: TShiftState; X, Y: Integer);
var row, col:integer;
begin

  if CONNECTED and (Button=mbLeft) and (ssAlt in Shift) and (PB.idx=0) then
  begin
    if sender=CSR then begin
                         row:=Ypos;
                         col:=min(Xpos, COLS)          // fix: can be =81 if at end of line
                       end
                  else begin
                         row:=trunc(((Y-Image1.Top)/CSR.Height)+1);
                         col:=trunc(((X-Image1.Left)/CSR.Width)+1);
                         if row<1 then row:=1;
                         if col<1 then col:=1;
                         if row>ROWS then row:=ROWS;
                         if col>COLS then col:=COLS
                       end;

    WriteComm(#27+'['+IntToStr(row)+';'+IntToStr(col)+'R')
  end
(*
    dY:=row-Ypos;
    dX:=col-Xpos;

//  ShowMessage('('+IntToStr(dX)+','+IntToStr(dY)+')');
    if dY<>0 then if dY<0 then CursorBuffer:='1'+StringOfChar('A', abs(dY))+'4'
                          else CursorBuffer:='1'+StringOfChar('B', abs(dY))+'4'

//  if dX<>0 then if dX<0 then for I:=1 to abs(dX) do WriteBuffer:=Writebuffer+#27+'[D'
//                        else for I:=1 to abs(dX) do WriteBuffer:=Writebuffer+#27+'[C'
//  ShowMessage('('+IntToStr(col)+','+IntToStr(row)+')');
  end
*)
end;


procedure TForm1.FormMouseMove(Sender: TObject; Shift: TShiftState; X,
  Y: Integer);
begin
  if Sender=Form1 then begin
                         MouseX:=X-Image1.Left;
                         MouseY:=Y-Image1.Top
                       end
                  else begin
                         MouseX:=-1;
                         MouseY:=-1
                       end
end;


////////////////////////////////////////////////////////////////////////////////
// startup (FormCreate) and shutdown (FormClose) code
// ==================================================
////////////////////////////////////////////////////////////////////////////////

procedure TForm1.FormCreate(Sender: TObject);
var Reg:TRegistry;
      I:integer;
begin
//Application.HintHidePause:=5000;
  Form1.Caption:='Graphical VT Terminal  (build: 8-September-2017)';
  Form1.AutoSize:=true;
//Form1.AutoScroll:=false;             // set in form view
//Form1.KeyPreview:=true;              // set in form view
  Form1.DoubleBuffered:=true;

  CSR:=Cursor;                         // CSR is shorthand for the cursor object

  Reg:=TRegistry.Create;               // try to read font size from registry
  with Reg do
  try
    RootKey:=HKEY_CURRENT_USER;
    if OpenKey(SoftwareKey, false) then
    try
      I:=ReadInteger('FontSize');
      if I in [9, 12, 14] then CSR.Font.Size:=I
    except
    end;
    try
      I:=ReadInteger('DefaultFG');
      if I in [1..7] then begin FGdefault:=I; FGcolour:=I end
    except
    end

  finally
    CloseKey;
    Reg.Free
  end;


  Image1.Left:=0;                      // horizontal position of screen grid
  Image1.Top:=Label1.Height+2;         // vertical position of screen grid
  Image1.Width:=CSR.Width*COLS;        // set width
  Image1.Height:=CSR.Height*ROWS;      // set height

  Image2.Left:=Image1.Left;            // graphics plane overlays text plane
  Image2.Top:=Image1.Top;
  Image2.Width:=Image1.Width;
  Image2.Height:=Image1.Height;

  Memo1.Left:=Image1.Left;             // text copy object
  Memo1.Top:=Image1.Top;
  Memo1.Width:=Image1.Width;
  Memo1.Height:=Image1.Height;
  Memo1.Font:=CSR.Font;
  Memo1.Font.Color:=clBlack;
  Memo1.Font.Style:=[];
  Memo1.Color:=clWhite;
  Memo1.Hint:='Select the text you want to copy using the mouse,'+#13+
              'then press control-C to copy it to the clipboard.'+#13+
              'When finished, press ENTER to exit this view.';

  Label7.Color:=clBtnFace;             // ring buffer usage, no special highlight
  Label8.Color:=clGreen;               // green 'running' annunciator by default
  Label9.Color:=clYellow;              // yellow 'logging' annunciator by default

  OpenDialog1.InitialDir:=ExtractFilePath(ExpandFileName(paramstr(0)));
  SaveDialog1.InitialDir:=ExtractFilePath(ExpandFileName(paramstr(0)));

// create text area
  Image1.Picture.Graphic:=TBitmap.Create;          // create a bitmap object for text
  Image1.Picture.Graphic.Width:=Image1.Width;      // set width
  Image1.Picture.Graphic.Height:=Image1.Height;    // set height

// create graphics area
  Image2.Transparent := TRUE;                          // allow transparence
  Image2.Picture.Graphic:=TBitmap.Create;              // create a bitmap object for graphics
  Image2.Picture.Graphic.Width:=Image2.Width;          // set width
  Image2.Picture.Graphic.Height:=Image2.Height;        // set height
  Image2.Picture.Bitmap.TransparentColor:=clBlack;     // transparent colour is black
  Image2.Picture.Bitmap.TransparentMode:=tmFixed;      // use above setting for TC

// align graphics and text layers along the Z-axis
  Memo1.Enabled:=false;
  Memo1.Visible:=false;
  Memo1.SendToBack;
  Image1.Enabled:=false;               // ignore mouse and keyboard events
  Image2.Enabled:=false;               // ignore mouse and keyboard events
  Image1.BringToFront;                 // text layer ends in background
  Image2.BringToFront;                 // graphics layer overlays text
  Cursor.BringToFront;                 // cursor always on top of text and graphics

  CSR.Top:=Image1.Top;                 // initial cursor row
  CSR.Left:=Image1.Left;               // initial cursor column
  CSR.Font.Color:=clRed;               // initial cursor colour (FG)
  CSR.Color:=clRed;                    // initisl cursor colour (BG)

  SCR:=Image1.Picture.Bitmap.Canvas;   // shorthand for canvas object (text screen)
  SCR.Font:=Cursor.Font;               // copy font details from cursor object
  SCR.TextFlags:=ETO_OPAQUE;           // opaque writing of text, improves speed
//SCR.Font.Style:=[];                  // no underline, etc.
//SCR.Brush.Color:=clBlack;            // default brush: clBlack (for clearing screen)
//SCR.Pen.Color:=clWhite;              // default pen: clWhite (for writing text)

  GFX:=Image2.Picture.Bitmap.Canvas;   // shorthand for canvas object (graphics screen)
  GFX.Brush.Color:=clBlack;            // default brush: clBlack (for clearing screen)
  GFX.Pen.Color:=clRed;                // default pen: clRed (for drawing lines)

  clear(1, 1, COLS, ROWS);             // clear text layer
  GFXclear(0, 0, Gw, Gh);              // clear graphics layer
  gotoxy(1,1);                         // home cursor

  TC1:=GetTickCount;
  TC2:=GetTickCount;
  TC3:=GetTickCount
end;


procedure TForm1.FormClose(Sender: TObject; var Action: TCloseAction);
begin
  if CONNECTED then try CloseHandle(CommFile) except end;
  if LOGTOFILE then try CloseFile(Logfile) except end;

  sleep(200)
end;


////////////////////////////////////////////////////////////////////////////////
// start: unused launch application code ///////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////
(*
procedure CaptureConsoleOutput(const ACommand, AParameters: string;
                               var ExitCode: integer);
var saSecurity:TSecurityAttributes;
    suiStartup:TStartupInfo;
     piProcess:TProcessInformation;
    ReadBuffer:array [0..127] of char;
     BytesRead,
    BytesAvail:DWORD;
       Running:DWORD;
           I,J:integer;
             S:string;
            ch:char;
            LB:LongBool;
 begin
   ExitCode:=-($DEAD);         // default exit code if unable to complete

   FillChar(saSecurity, SizeOf(TSecurityAttributes), #0);
   saSecurity.nLength:=SizeOf(TSecurityAttributes);
   saSecurity.bInheritHandle:=true;            // seems to be essential for pipes
   saSecurity.lpSecurityDescriptor:=nil;

   if CreatePipe(hRead_local, hWrite_remote, @saSecurity, 0) then
   if CreatePipe(hRead_remote, hWrite_local, @saSecurity, 0) then     { try 1 byte write }
   begin
     FillChar(suiStartup, SizeOf(TStartupInfo), #0);
     suiStartup.cb:=SizeOf(TStartupInfo);
     suiStartup.hStdInput:=hRead_remote;
     suiStartup.hStdOutput:=hWrite_remote;
     suiStartup.hStdError:=hWrite_remote;
     suiStartup.dwFlags:=STARTF_USESTDHANDLES or STARTF_USESHOWWINDOW;
     suiStartup.wShowWindow:=SW_HIDE;
{
   dwXCountChars, dwYCountChars
dwFlags specifies STARTF_USECOUNTCHARS.
For console processes,
if a new console window is created,
dwXCountChars specifies the screen buffer
width in character columns, and dwYCountChars
specifies the screen buffer height in character rows.
These values are ignored in GUI processes.  }

     saSecurity.bInheritHandle:=false;         // standard handles are still inherited

     if CreateProcess(nil, PChar(ACommand + ' ' + AParameters),
                      @saSecurity, @saSecurity,
//                    nil, nil,
                      True, NORMAL_PRIORITY_CLASS,
                      nil, nil, suiStartup, piProcess) then
     begin
       CONNECTED:=true;
// ***************************************
//     try CloseHandle(hWrite) except ShowMessage('unable to close hWrite') end;
// ***************************************

       repeat
         Running:=WaitForSingleObject(piProcess.hProcess, 20);
//       Application.ProcessMessages();
//       inc(LC1);
         repeat
//         inc(LC2);
           Application.ProcessMessages();
           try
             if PeekNamedPipe(hRead_local, nil, 0, nil, @BytesAvail, nil) and (BytesAvail>0)
                then LB:=ReadFile(hRead_local, ReadBuffer[0],
                                  min(sizeof(ReadBuffer), BytesAvail), BytesRead, nil)
                else LB:=false
           except
             LB:=false
           end;       // may exit with ERROR_BROKEN_PIPE after child process closes

           if not LB then BytesRead:=0 else
           begin
             for I:=1 to BytesRead do
             begin
               ch:=ReadBuffer[I-1];
               S:=VT100engine(ch);
               for J:=1 to length(S) do emit(S[J])
             end
           end;
         until (BytesRead<sizeof(ReadBuffer)) or KILL
       until (Running<>WAIT_TIMEOUT) or KILL;

       CONNECTED:=false;

       GetExitCodeProcess(piProcess.hProcess, DWORD(ExitCode));  // returns STILL_ACTIVE if still running
       if KILL then TerminateProcess(piProcess.hProcess, 0);

       try CloseHandle(piProcess.hProcess) except end;
       try CloseHandle(piProcess.hThread) except end
     end;

     try CloseHandle(hRead_local) except end;
     try CloseHandle(hWrite_local) except end;
     try CloseHandle(hRead_remote) except end;
     try CloseHandle(hWrite_remote) except end

   end;
   DEAD:=true
end;


procedure RUNprogram;
var I:integer;
    S:string;
begin
  if not CONNECTED then
  if inputquery('RUN', 'enter program to launch', S) then
  begin
    DEAD:=false;
    KILL:=false;
    emit(#13);
    emit(#10);
    CaptureConsoleOutput(S,'', I);
    emit(#13);
    emit(#10)
  end
end;


procedure STOPprogram;
begin
  KILL:=true
end;
*)
////////////////////////////////////////////////////////////////////////////////
// end: unused launch application code /////////////////////////////////////////
////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
// popup menu item handlers
// ========================
////////////////////////////////////////////////////////////////////////////////

// *** popup menu - configure menu before making visible ***********************
procedure TForm1.PopupMenu1Popup(Sender: TObject);
begin
  Item1A.Visible:=not CONNECTED;
  Item1B.Visible:=CONNECTED;
  Item2A.Visible:=not LOGTOFILE;
  Item2B.Visible:=LOGTOFILE;
  Item3A.Visible:=(PB.idx=0);
  Item3B.Visible:=(PB.idx<>0);
  case CSR.Font.Size of  9:Item4A.Checked:=true;
                        12:Item4B.Checked:=true;
                        14:Item4C.Checked:=true
  end;  {of case }
  Case FGdefault of 1:Item5A.Checked:=true;
                    2:Item5B.Checked:=true;
                    3:Item5C.Checked:=true;
                    4:Item5D.Checked:=true;
                    5:Item5E.Checked:=true;
                    6:Item5F.Checked:=true;
                    7:Item5G.Checked:=true
  end  { of case }
end;


// popup menu item - CONNECT / DISCONNECT
procedure TForm1.Item1Click(Sender: TObject);
const init:boolean=true;
         S:string='';
var name,config:str255;
             ok:boolean;
              I:integer;
            Reg:TRegistry;
begin
  if init then begin                           // try to read S from registry
                  Reg:=TRegistry.Create;
                  with Reg do
                  try
                    RootKey:=HKEY_CURRENT_USER;
                    if OpenKey(SoftwareKey, false) then
                      try S:=ReadString('CommPort') except S:='' end
                  finally
                    CloseKey;
                    Reg.Free
                  end
               end;

  if CONNECTED then begin
                      try CloseHandle(CommFile) except end;
                      CONNECTED:=false
                    end
               else repeat
                      ok:= InputQuery('CONNECT', 'enter <port>:<baud>', S);
                      if ok then
                      begin
                        I:=pos(':',S);

                        if I<>0 then
                        begin
                          name:=copy(S,1,I-1);
                          config:='baud='+copy(S,I+1,length(S)-I)+' parity=n data=8 stop=1';
                          SetupCommPort(name, config)
                        end
                      end;

                      if CONNECTED then        // save S into registry
                      begin
                        Reg:=TRegistry.Create;
                        with Reg do
                        try
                          RootKey:=HKEY_CURRENT_USER;
                          if OpenKey(SoftwareKey,  true) then
                            try WriteString('CommPort', S) except end
                        finally
                          CloseKey;
                          Reg.Free
                        end
                      end
                    until CONNECTED or not ok
end;


// popup menu item - LOG to file / STOP logging
procedure TForm1.Item2Click(Sender: TObject);
begin
  if LOGTOFILE then begin
                      try CloseFile(LogFile) except end;
                      LOGTOFILE:=false
                    end
               else begin
                      if SaveDialog1.Execute then
                      try
                        AssignFile(LogFile, SaveDialog1.Filename);
                        ReWrite(LogFile);
                        LOGTOFILE:=true
                      except
                        try CloseFile(Logfile) except end;
                        LOGTOFILE:=false
                      end
                    end
end;


// popup menu item - paste from clipboard / paste from text file / CANCEL paste
procedure TForm1.Item3Click(Sender: TObject);
var SL:TStringList;
     S:string;
     I:integer;
begin
// setup for paste from clipboard
  if Sender=Item3A1 then begin
                           if ClipBoard.HasFormat(CF_TEXT) then
                           try
                             S:=ClipBoard.AsText;
                           except
                             S:=''
                           end
                         end;

// setup for paste from text file
  if Sender=Item3A2 then begin
                           if OpenDialog1.Execute then
                           begin
                             SL:=TStringlist.Create;
                             try
                               SL.LoadFromFile(OpenDialog1.Filename);
                               S:=SL.Text;
                             except
                               S:=''
                             end;
                             SL.Free
                           end
                         end;

// common code for both types of paste (from clipboard and from text file)
  if (Sender=Item3A1) or (Sender=Item3A2) then
  begin
    I:=pos(#13#10, S);
    while I<>0 do
    begin                                // translate all CR-LF pairs into single CR
      delete(S, I+1, 1);
      I:=pos(#13#10, S)
    end;

    for I:=1 to length(S) do             // translate any remaining LFs into CRs
        if S[I]=#10 then S[I]:=#13;

    for I:=length(S) downto 1 do         // remove any other control characters
        if S[I] in [#0..#12,#14..#31] then delete(S, I, 1);

    I:=pos(#32#13, S);
    while I<>0 do
    begin                                // remove all trailing spaces at ends of lines
      delete(S, I, 1);
      I:=pos(#32#13, S)
    end;

    while (length(S)<>0) and (S[length(S)]=#32) do delete(S, length(S), 1);
                                         // remove final trailing spaces at end of S
    if length(S)<>0 then
    begin
      PB.str:=S;                         // place into string to be streamed out
      PB.len:=length(S);                 // the comm port by a timer interrupt
      PB.idx:=1
    end
  end;

// code for CACEL paste
  if Sender=Item3B then begin            // cancel paste
                          PB.idx:=0;
                          PB.len:=0;
                          PB.str:=''
                        end
end;


// popup menu item - font size 9pt / 12pt / 14pt
procedure TForm1.Item4Click(Sender: TObject);
var Reg:TRegistry;
begin
  if Sender=Item4A then CSR.Font.Size:=9 else      // select 9pt text size
  if Sender=Item4B then CSR.Font.Size:=12 else     // select 12pt text size
  if Sender=Item4C then CSR.Font.Size:=14;         // select 14pt text size

  Image1.Width:=CSR.Width*COLS;                    // set width
  Image1.Height:=CSR.Height*ROWS;                  // set height

  Image2.Width:=Image1.Width;
  Image2.Height:=Image1.Height;

  Memo1.Width:=Image1.Width;
  Memo1.Height:=Image1.Height;
  Memo1.Font:=CSR.Font;
  Memo1.Font.Color:=clBlack;
  Memo1.Font.Style:=[];

  Image1.Picture.Graphic.Width:=Image1.Width;      // set width
  Image1.Picture.Graphic.Height:=Image1.Height;    // set height

  Image2.Picture.Graphic.Width:=Image2.Width;      // set width
  Image2.Picture.Graphic.Height:=Image2.Height;    // set height

  SCR.Font:=Cursor.Font;                           // copy font details from cursor

  clear(1, 1, COLS, ROWS);                         // clear text layer
  GFXclear(0, 0, Gw, Gh);                          // clear graphics layer
  gotoxy(1,1);                                     // home cursor

  Reg:=TRegistry.Create;
  with Reg do                          // save font size into registry
  try
    RootKey:=HKEY_CURRENT_USER;
    if OpenKey(SoftwareKey, true) then
      try WriteInteger('FontSize', CSR.Font.Size) except end
  finally
    CloseKey;
    Reg.Free
  end
end;


// popup menu item - select default text colour
procedure TForm1.Item5Click(Sender: TObject);
var Reg:TRegistry;
     TC:integer;
begin
  if Sender=Item5A then TC:=1 else
  if Sender=Item5B then TC:=2 else
  if Sender=Item5C then TC:=3 else
  if Sender=Item5D then TC:=4 else
  if Sender=Item5E then TC:=5 else
  if Sender=Item5F then TC:=6 else
  if Sender=Item5G then TC:=7
                   else TC:=-1;
  if TC<>-1 then begin FGdefault:=TC; FGcolour:=TC end;

  Reg:=TRegistry.Create;
  with Reg do                          // save default FG colour into registry
  try
    RootKey:=HKEY_CURRENT_USER;
    if OpenKey(SoftwareKey, true) then
      try WriteInteger('DefaultFG', FGdefault) except end
  finally
    CloseKey;
    Reg.Free
  end
end;


// popup menu item - clear GFX plane / text plane / ring buffer
procedure TForm1.Item6Click(Sender: TObject);
begin
  if Sender=Item6A then GFXclear(0, 0, Gw, Gh) else    // clear graphics layer
  if Sender=Item6B then clear(1, 1, COLS, ROWS) else   // clear text layer
  if Sender=Item6C then RB.head:=RB.tail               // clear ring buffer
end;


// popup menu item - show copy memo
procedure TForm1.Item7Click(Sender: TObject);
var X, Y:integer;
       S:string;
begin
  Memo1.Lines.Clear;
  for Y:=1 to ROWS do                          // fill in memo text
  begin
    S:=StringOfChar(' ', COLS);                // blank line of 80 chars
    for X:=1 to COLS do S[X]:=TextStore[Y, X]; // copy text from buffer (TextStore)
    S:=TrimRight(S);
    Memo1.Lines.Add(S)                         // transfer completed line into memo
  end;

  S:=Memo1.Text;
  S:=copy(S, 1, length(S)-2);                  // remove final CR-LF from memo
  Memo1.Text:=S;

  Form1.PopupMenu:=nil;                        // disable our popup menu
                                               // (cut/copy menu takes over)
  Memo1.Enabled:=true;                         // probably not needed
  Memo1.Visible:=true;
//Memo1.BringToFront;                          // probably not needed
//Memo1.SetFocus                               // probably not needed
end;


// popup menu item - make command window visible
procedure TForm1.Item8Click(Sender: TObject);
begin
  Form2.Visible:=true;
  Form2.BringToFront
end;




end.


